//*************************************************************************************************
//
//	Description:
//		windows.fx - Specialisation of the basic object shader which supports time-of-day variation
//								 for building windows.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/27/2006  0.1           Created
//		TNettleship     05/12/2006  1.0           First "offical" release to FTP
//		TNettleship     05/18/2006  1.1           Added technique annotations.
//		TNettleship     06/09/2006  1.2           Added ambient occlusion term from vertex colour,
//																							global emap factor, removed ambient light, added
//																							option for compressed normals.
//		TNettleship			09/01/2006	1.3						Rewrote the window lighting function.
//		TNettleship			10/03/2006	1.4						Fixed a bug with channel assignment & vertex colours
//																							in 3DSMAX.
//		TNettleship			10/18/2006	1.5						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.6						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			10/25/2006	1.7						Changed binormals from a calculation back to an input.
//		TNettleship			11/02/2006	1.8						Fixed a bug with directional lighting.
//		TMann						11/23/2006	1.10					Added _WIN_GL_ support
//		TMann						11/27/2006	1.11					Texture filtering for GL
//		TMann						12/11/2006	1.13					PS3 version
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//		TNettleship			10/23/2007	2.04					Converted to on-load rendermode behaviour binding.
//		TNettleship			11/23/2007 2.05					Modifications to support prebaked lighting.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//




//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Textures
//

//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

#endif


#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

#define MAX_DIGITS		8
float digitArray[MAX_DIGITS]
<
	string UIName = "digitArray";
	bool appEdit = true;
	bool export = true;
>; 


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };

//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float4 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 2 texture coord
#else
	float4 colour    : COLOR0;														// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord
	float2 texCoord1 : TEXCOORD1;													// UV channel 2 texture coord
#endif
	float3 normal    : NORMAL;														// Object space normal
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 4 )
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT OdometerVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate vert's world coords normal
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;

	//Calculate digit texcoord - digitArray contains v offset per digit
	//digit index in 2nd uv channel
	float2 tex0 = _input.texCoord0;
	float voffset = digitArray[ _input.texCoord1.x ];
	_output.texCoord0 = float2( tex0.x, tex0.y + voffset );

	// Calculate object-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1;												// Normal vector (world space)
	float4 eye				: TEXCOORD2;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 4 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float3 normal			: TEXCOORD1_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD2_centroid;								// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS( 4 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT OdometerFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord0 );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord0 );

  // Create tangent space vectors
	float3 normal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );

	// Calculate ambient colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = 1.0f;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Odometer
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Odometer";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx OdometerVertexShader();
		PixelShader = compile sce_fp_rsx OdometerFragmentShader();
#else
		VertexShader = compile vs_3_0 OdometerVertexShader();
		PixelShader = compile ps_3_0 OdometerFragmentShader();
#endif		
	}
}
